#!/usr/bin/env python3
"""
Standardized Python runner template for After Effects batch rendering.
All runners should follow this structure for consistency.
"""
import os, subprocess, shlex, tempfile

# ============================================================================
# CONFIGURATION - Modify these values for each animation
# ============================================================================

AFTER_EFFECTS_APP = "Adobe After Effects 2025"

# Project paths
PROJECT   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/YourProject.aep"
CSV_PATH  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
JSX_BATCH = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_YourScript.jsx"
COMMON_JSX= "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/gl_common.jsxinc"

# After Effects composition settings
COMP        = "Comp 1"
ANIM_NAME   = "ANIM_1_3"  # Animation identifier (e.g., TD_2_6, FG_1_3, HR_1_1)

# Output settings
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Output_Folder"
PATH_TEMPLATE = "{league}"  # Output path template: {league}, {league}/{abbr}, etc.
OM_TEMPLATE   = "PNG Sequence"  # Output Module template: "PNG Sequence" or "H.264 - Match Source - High Bitrate"
EXT           = ""  # File extension for video outputs (e.g., ".mp4"), leave empty for PNG sequences

# Run mode
RUN_MODE = "league"      # "league" or "subset"
LEAGUE   = "NFL"        # Target league (NFL, MLB, NBA, NHL, CFB, etc.)
SUBSET   = ["BUF","DAL","PHI"]  # Only used if RUN_MODE == "subset"

# Layer names (optional - override JSX defaults if needed)
LAYER_OVERRIDES = {
    # Example overrides:
    # "AE_TEXT_STROKE_LAYER": "TOUCHDOWN",
    # "AE_TEAMNAME_LAYER": "TeamName",
    # "AE_LOGO_LAYER": "Logo",
}

# Logo settings
LOGO_DIR = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large"
LOGO_PATH_TEMPLATE = "{league}/{abbr}"
LOGO_EXTS = "png,jpg,jpeg,svg,ai,psd"

# ============================================================================
# HELPER FUNCTIONS - Do not modify below this line
# ============================================================================

def _write_bridge_jsx(env):
    """Create temporary JSX bridge file that loads environment variables"""
    def esc(s): return s.replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(str(v))+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines)+"\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False, mode='w')
    tmp.write(code)
    tmp.flush()
    tmp.close()
    return tmp.name

def run(env):
    """Execute After Effects with environment variables"""
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa = [
        "osascript",
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to activate',
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to DoScriptFile POSIX file "{bridge}"',
    ]
    print("Running:", " ".join(shlex.quote(c) for c in osa))
    subprocess.run(osa, check=True)
    print("\nDone. Output ->", OUTDIR)

def make_env(league=None, teams=None):
    """Build environment variable dictionary for After Effects"""
    env = {
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMMON_JSX": COMMON_JSX,
        "AE_COMP": COMP,

        "AE_LOGO_DIR": LOGO_DIR,
        "AE_LOGO_PATH_TEMPLATE": LOGO_PATH_TEMPLATE,
        "AE_LOGO_EXTS": LOGO_EXTS,

        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": ANIM_NAME,
        "AE_RS_TEMPLATE": "Best Settings",
        "AE_OM_TEMPLATE": OM_TEMPLATE,
        "AE_EXT": EXT,

        "AE_PURGE_BEFORE_RENDER": "1",
        "AE_NO_RENDER": "0",
        "AE_QUIT": "1",
    }

    # Add layer overrides
    env.update(LAYER_OVERRIDES)

    # Add league/team filters
    if league: env["AE_LEAGUE"] = league
    if teams:  env["AE_TEAMS"]  = ",".join(teams)

    return env

if __name__ == "__main__":
    # Validate files exist
    assert os.path.isfile(JSX_BATCH), f"JSX not found: {JSX_BATCH}"
    assert os.path.isfile(PROJECT),   f"Project not found: {PROJECT}"
    assert os.path.isfile(CSV_PATH),  f"CSV not found: {CSV_PATH}"
    assert os.path.isfile(COMMON_JSX),f"Common JSX not found: {COMMON_JSX}"

    # Run based on mode
    if RUN_MODE == "league":
        run(make_env(league=LEAGUE))
    else:
        run(make_env(teams=SUBSET))
